program BDDVALUE;
{--------------------------------------------------------------------}
{  Alg9'910.pas   Pascal program for implementing Algorithm 9.9-10   }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 9.9 (Linear Shooting Method).                           }
{  Section   9.8, Boundary Value Problems, Page 488                  }
{                                                                    }
{  Algorithm 9.10 (Finite-Difference Method).                        }
{  Section   9.9,  Finite-Difference Method, Page 496                }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    GNmax = 630;
    MaxM = 101;
    FunMax = 9;

  type
    VECTOR = array[0..MaxM] of real;
    DVECTOR = array[1..4] of real;
    LETTER = string[8];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);
    LETTERS = string[200];

  var
    FunType, GNpts, Inum, M, Mend, Meth, Order, Sub: integer;
    A, B, Alpha, Beta, Rnum, Y0: real;
    Ans: CHAR;
    T, X: VECTOR;
    D: DVECTOR;
    State: States;
    DoMo: DoSome;
    Mess: LETTERS;

  function P (T: real): real;
  begin
    case FunType of
      1: 
        P := 2 * T / (1 + T * T);
      2: 
        P := -2 / T;
      3: 
        P := -5;
      4: 
        P := -2;
      5: 
        P := -1 / T;
      6: 
        P := -2 / T;
      7: 
        P := -1 / T;
      8: 
        P := 1 / T;
      9: 
        P := -1 / T;
    end;
  end;

  function Q (T: real): real;
  begin
    case FunType of
      1:
        Q := -2 / (1 + T * T);
      2: 
        Q := 2 / (T * T);
      3: 
        Q := -6;
      4: 
        Q := -2;
      5: 
        Q := -(1 - 1 / (4 * T * T));
      6: 
        Q := 2 / (T * T);
      7: 
        Q := -(1 - 1 / (4 * T * T));
      8: 
        Q := -1 / (T * T);
      9: 
        Q := -16 / (T * T);
    end;
  end;

  function R (T: real): real;
  begin
    case FunType of
      1: 
        R := 1;
      2: 
        R := 10 * COS(LN(T)) / (T * T);
      3: 
        R := T * EXP(-2 * T) + 3.9 * COS(3 * T);
      4: 
        R := EXP(-T) + SIN(2 * T);
      5: 
        R := 0;
      6:
        R := SIN(T) / (T * T);
      7: 
        R := SQRT(T) * COS(T);
      8: 
        R := 1;
      9: 
        R := 1 / (T * T);
    end;
  end;

  function F1 (T, X, Y: real): real;
  begin
    case FunType of
      1: 
        F1 := 2 * T / (1 + T * T) * Y - 2 / (1 + T * T) * X + 1;
      2: 
        F1 := -2 / T * Y + 2 / (T * T) * X + 10 * COS(LN(T)) / (T * T);
      3: 
        F1 := -5 * Y - 6 * X + T * EXP(-2 * T) + 3.9 * COS(3 * T);
      4: 
        F1 := -2 * Y - 2 * X + EXP(-T) + SIN(2 * T);
      5: 
        F1 := -1 / T * Y - (1 - 1 / (4 * T * T)) * X;
      6: 
        F1 := -2 / T * Y + 2 / (T * T) * X + SIN(T) / (T * T);
      7: 
        F1 := -1 / T * Y - (1 - 1 / (4 * T * T)) * X + SQRT(T) * COS(T);
      8: 
        F1 := 1 / T * Y - 1 / (T * T) * X + 1;
      9: 
        F1 := -1 / T * Y - 16 / (T * T) * X + 1 / (T * T);
    end;
  end;

  function F2 (T, X, Y: real): real;
  begin
    case FunType of
      1: 
        F2 := 2 * T / (1 + T * T) * Y - 2 / (1 + T * T) * X;
      2: 
        F2 := -2 / T * Y + 2 / (T * T) * X;
      3: 
        F2 := -5 * Y - 6 * X;
      4: 
        F2 := -2 * Y - 2 * X;
      5: 
        F2 := -1 / T * Y - (1 - 1 / (4 * T * T)) * X;
      6: 
        F2 := -2 / T * Y + 2 / (T * T) * X;
      7: 
        F2 := -1 / T * Y - (1 - 1 / (4 * T * T)) * X;
      8: 
        F2 := 1 / T * Y - 1 / (T * T) * X;
      9: 
        F2 := -1 / T * Y - 16 / (T * T) * X;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1:
        WRITELN('X`` =  2*T*X`/(1 + T*T) - 2*X/(1 + T*T) + 1');
      2:
        WRITELN('X`` =  -2/T*X` + 2/(T*T) * X + 10*COS(LOG(T))/(T*T)');
      3:
        WRITELN('X`` =  -5*X` - 6*X + T * EXP(-2 * T) + 3.9 * COS(3*T)');
      4:
        WRITELN('X`` =  -2*X` - 2*X + EXP(-T) + SIN(2 * T)');
      5:
        WRITELN('X`` =  -1/T*X` - (1 - 1 / (4*T*T)) * X');
      6:
        WRITELN('X`` =  -2/T * X` + 2/(T*T) * X + SIN(T)/(T*T)');
      7:
        WRITELN('X`` =  -1/T * X` - (1 - 1 / (4*T*T)) * X + SQRT(T) * COS(T)');
      8:
        WRITELN('X`` =  1/T * X` - 1/(T*T) * X + 1');
      9:
        WRITELN('X`` =   -1/T * X` - 16/(T*T) * X + 1/(T*T)');
    end;
  end;

  procedure LinearShooting ({FUNCTION F1(t,x,y:real),F2(t,x,y:real): real;}
                  A, B, Alpha, Beta: real; M: integer; var T, X: VECTOR);
    var
      J: integer;
      X1, X2: VECTOR;

    procedure RKbdd4forF1 ({FUNCTION F1(t,x,y:real): real;}
                    A, B, Alpha, Beta: real; M: integer; var T, X: VECTOR);
      var
        J: integer;
        H, K1, K2, K3, K4, R1, R2, R3, R4, Tj, Xj, Yj: real;
        Y: Vector;
    begin
      H := (B - A) / M;
      T[0] := A;
      X[0] := Alpha;
      Y[0] := Beta;
      for J := 0 to M - 1 do
        begin
          Tj := T[J];
          Xj := X[J];
          Yj := Y[J];
          K1 := H * Yj;
          R1 := H * F1(Tj, Xj, Yj);
          K2 := H * (Yj + R1 / 2);
          R2 := H * F1(Tj + H / 2, Xj + K1 / 2, Yj + R1 / 2);
          K3 := H * (Yj + R2 / 2);
          R3 := H * F1(Tj + H / 2, Xj + K2 / 2, Yj + R2 / 2);
          K4 := H * (Yj + R3);
          R4 := H * F1(Tj + H, Xj + K3, Yj + R3);
          X[J + 1] := Xj + (K1 + 2 * K2 + 2 * K3 + K4) / 6;
          Y[J + 1] := Yj + (R1 + 2 * R2 + 2 * R3 + R4) / 6;
          T[J + 1] := A + H * (J + 1);
        end;
    end;

    procedure RKbdd4forF2 ({FUNCTION F2(t,x,y:real): real;}
                    A, B, Alpha, Beta: real; M: integer; var T, X: VECTOR);
      var
        J: integer;
        H, K1, K2, K3, K4, R1, R2, R3, R4, Tj, Xj, Yj: real;
        Y: Vector;
    begin
      H := (B - A) / M;
      T[0] := A;
      X[0] := Alpha;
      Y[0] := Beta;
      for J := 0 to M - 1 do
        begin
          Tj := T[J];
          Xj := X[J];
          Yj := Y[J];
          K1 := H * Yj;
          R1 := H * F2(Tj, Xj, Yj);
          K2 := H * (Yj + R1 / 2);
          R2 := H * F2(Tj + H / 2, Xj + K1 / 2, Yj + R1 / 2);
          K3 := H * (Yj + R2 / 2);
          R3 := H * F2(Tj + H / 2, Xj + K2 / 2, Yj + R2 / 2);
          K4 := H * (Yj + R3);
          R4 := H * F2(Tj + H, Xj + K3, Yj + R3);
          X[J + 1] := Xj + (K1 + 2 * K2 + 2 * K3 + K4) / 6;
          Y[J + 1] := Yj + (R1 + 2 * R2 + 2 * R3 + R4) / 6;
          T[J + 1] := A + H * (J + 1);
        end;
    end;

  begin                                {The main program LinearShooting}
    RKbdd4forF1(A, B, Alpha, 0, M, T, X1);
    RKbdd4forF2(A, B, 0, 1, M, T, X2);
    for J := 0 to M do
      X[J] := X1[J] + (Beta - X1[M]) * X2[J] / X2[M];
  end;

  procedure FiniteDiff ({FUNCTION P(t:real), Q(t:real), R(t:real): real;}
                  A, B, Alpha, Beta: real; N: integer; var T, X: VECTOR);
    var
      H: real;
      Va, Vb, Vc, Vd: VECTOR;           {Remark: Some of the zero components are NOT used.}

    procedure CoeffMat (A, B, Alpha, Beta: real; N: integer; var Va, Vb, Vc, Vd, Vt: VECTOR);
      var
        j: integer;
    begin
      H := (B - A) / N;
      for j := 1 to N - 1 do
        Vt[j] := A + H * j;
      for j := 1 to N - 1 do
        Vb[j] := -H * H * R(Vt[j]);
      Vb[1] := Vb[1] + (1 + H / 2 * P(Vt[1])) * Alpha;
      Vb[N - 1] := Vb[N - 1] + (1 - H / 2 * P(Vt[N - 1])) * Beta;
      for j := 1 to N - 1 do
        Vd[j] := 2 + H * H * Q(Vt[j]);
      for j := 1 to N - 2 do
        Va[j] := -1 - H / 2 * P(Vt[j + 1]);
      for j := 1 to N - 2 do
        Vc[j] := -1 + H / 2 * P(Vt[j]);
    end;

    procedure TriMat (Va, Vb, Vc, Vd: VECTOR; N: integer; var X: VECTOR);
      var
        k: integer;
        tem: real;
    begin
      for k := 2 to N - 1 do
        begin
          tem := Va[k - 1] / Vd[k - 1];
          Vd[k] := Vd[k] - tem * Vc[k - 1];
          Vb[k] := Vb[k] - tem * Vb[k - 1]
        end;
      X[N - 1] := Vb[N - 1] / Vd[N - 1];
      for k := N - 2 downto 1 do
        X[k] := (Vb[k] - Vc[k] * X[k + 1]) / Vd[k];
    end;

  begin                                                      {The main program  FiniteDiff  starts here}
    CoeffMat(A, B, Alpha, Beta, N, Va, Vb, Vc, Vd, T);
    TriMat(Va, Vb, Vc, Vd, N, X);
    T[0] := A;
    T[N] := B;
    X[0] := Alpha;
    X[N] := Beta;
  end;

  procedure MESSAGE (var Meth: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('                      BOUNDARY VALUE DIFFERENTIAL EQUATIONS');
    WRITELN;
    WRITELN;
    WRITELN('          Solution of the differential equation');
    WRITELN;
    WRITELN('                     x`` = p(t)x ` ( t ) + q ( t ) x ( t ) + r ( t )');
    WRITELN;
    WRITELN('          with   x(a) = Alpha  and  x(b) = Beta  computed over  [a,b].');
    WRITELN;
    WRITELN;
    WRITELN('                      Choose the method of approximation:');
    WRITELN;
    WRITELN('                      < 1 >  Linear shooting method');
    WRITELN;
    WRITELN('                      < 2 >  Finite difference method');
    WRITELN;
    Mess := '                      SELECT < 1 - 2 > ?  ';
    Meth := 1;
    WRITE(Mess);
    READLN(Meth);
    if (Meth < 1) and (State <> Changes) then
      Meth := 1;
    if (Meth > 2) and (State <> Changes) then
      Meth := 2;
  end;

  procedure INPUT (var FunType: integer; var A, B, Y0: real; var M: integer; MaxM: integer);
    var
      K: integer;
      Ans: LETTER;
  begin
    CLRSCR;
    WRITELN;
    WRITE('       You chose ');
    case Meth of
      1: 
        WRITE('the linear shooting method');
      2: 
        WRITE('the finite difference method');
    end;
    WRITELN(' to solve Y` = F(T,Y).');
    WRITELN;
    WRITELN('                          Choose your D.E.:');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('  <', K : 1, '>  ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '                          SELECT < 1 - 9 > ?  ';
    FunType := 1;
    WRITE(Mess);
    READLN(FunType);
    if (FunType < 1) and (State <> Changes) then
      FunType := 1;
    if (FunType > FunMax) and (State <> Changes) then
      FunType := FunMax;
    CLRSCR;
    WRITELN;
    WRITE('          You chose ');
    case Meth of
      1: 
        WRITE('the linear shooting method');
      2: 
        WRITE('the finite difference method');
    end;
    WRITELN(' to solve the D.E.');
    WRITELN;
    WRITE('                    ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('          with the initial condition  X(A) = Alpha  and  X(B) = Beta.');
    WRITELN;
    WRITELN('          A numerical approximation is computed over [A,B].');
    WRITELN;
    WRITELN('          You must give the endpoints for the interval,');
    WRITELN;
    WRITELN('           the boundary values, and the number of steps.');
    WRITELN;
    WRITELN;
    WRITE('          Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure EPOINTS (var A, B, Y0: real; var M: integer; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      I: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        CLRSCR;
        WRITELN;
        WRITE('                 ');
        PRINTFUNCTION(FunType);
        WRITELN;
        WRITELN;
        if (Stat = Enter) then
          begin
            Mess := '     ENTER  the  left  endpoint  A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            Mess := '     ENTER  the  right endpoint  B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
            Mess := '     ENTER initial condition  X(A) = ';
            WRITE(Mess);
            READLN(Alpha);
            WRITELN;
            Mess := '     ENTER initial condition  X(B) = ';
            WRITE(Mess);
            READLN(Beta);
            WRITELN;
            Mess := '     ENTER the number of steps   M = ';
            M := 1;
            WRITE(Mess);
            READLN(M);
            WRITELN;
            if M < 1 then
              M := 1;
            if M > 1000 then
              M := 1000;
          end
        else
          begin
            WRITELN('     The  left  endpoint  is     A =', A : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The  right endpoint  is     B =', B : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     Initial   condition  is  X(A) =', Alpha : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     Initial   condition  is  X(B) =', Beta : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The number of steps  is     M =  ', M : 2);
          end;
        WRITELN;
        WRITELN;
        WRITE('     Want to make a change ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            CLRSCR;
            WRITELN;
            WRITE('                 ');
            PRINTFUNCTION(FunType);
            WRITELN;
            WRITELN('     The current left  endpoint is A =', A : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('     The current right endpoint is B =', B : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
            WRITELN('     The   current   B. V.  is  X(A) =', Alpha : 15 : 7);
            Mess := '     Now  ENTER the NEW  B. V.  X(A) = ';
            WRITE(Mess);
            READLN(Alpha);
            WRITELN;
            WRITELN('     The   current   B. V.  is  X(B) =', Beta : 15 : 7);
            Mess := '     Now  ENTER the NEW  B. V.  X(B) = ';
            WRITE(Mess);
            READLN(Beta);
            WRITELN;
            WRITELN('     The  current value of  M  is  M =  ', M : 2);
            Mess := '     Now  ENTER  the NEW value of  M = ';
            WRITE(Mess);
            READLN(M);
            if (M < 1) then
              M := 1;
            if (M > 1000) then
              M := 1000;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RESULTS (FunType: integer; T, X: VECTOR; M, Mend: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    case Meth of
      1: 
        WRITE('The linear shooting method');
      2: 
        WRITE('The finite difference method');
    end;
    WRITELN(' was used to solve the D.E.');
    WRITELN;
    WRITE('      ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('with  X(', T[0] : 15 : 7, '  ) =', X[0] : 15 : 7);
    WRITELN('and   X(', T[M] : 15 : 7, '  ) =', X[M] : 15 : 7);
    WRITELN;
    WRITELN('    K', '       T(K)          ', '         X(K)');
    WRITELN('  ------------------------------------------------');
    WRITELN;
    for K := 0 to M do
      begin
        WRITELN(K : 5, '   ', T[K] : 15 : 7, '     ', X[K] : 15 : 7);
        WRITELN;
        if K mod 11 = 9 then
          begin
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
            WRITELN;
          end;
      end;
    Mend := M;
    if Mend < M then
      begin
        WRITELN('The solution points are approaching a pole.');
        WRITELN;
      end;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  A := 0;
  B := 1;
  Y0 := 0;
  M := 1;
  State := Working;
  while Meth <> 0 do
    begin
      MESSAGE(Meth);
      DoMo := Go;
      while DoMo = Go do
        begin
          INPUT(FunType, A, B, Y0, M, MaxM);
          while (State = Working) or (State = Changes) do
            begin
              EPOINTS(A, B, Y0, M, State);
              case Meth of
                1: 
                  LinearShooting(A, B, Alpha, Beta, M, T, X);
                2: 
                  FiniteDiff(A, B, Alpha, Beta, M, T, X);
              end;
              RESULTS(FunType, T, X, M, Mend);
              WRITELN;
              WRITELN;
              WRITE('Want to use a  different  initial condition ?  <Y/N>  ');
              READLN(Ans);
              WRITELN;
              if (Ans <> 'Y') and (Ans <> 'y') then
                State := Done
              else
                State := Changes;
            end;
          WRITELN;
          WRITE('Want to  change  the  differential equation ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            DoMo := Stop
          else
            State := Changes;
        end;
      WRITELN;
      WRITE('Want to try another method of approximation ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Meth := 0
      else
        State := Changes;
    end;
end.                                            {End of Main Program}

